<?php
/**
 * Export to spreadsheet model file.
 *
 * @package Model
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

use App\Config;
use App\Db\Query;
use App\Export\Records;
use App\FieldCoordinatorTransformer\QueryGeneratorFieldTransformer;
use App\Fields\Currency;
use App\Json;
use PhpOffice\PhpSpreadsheet\Cell\DataType;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;

/**
 * Export to spreadsheet model class.
 */
class Vtiger_ExportToSpreadsheet_Model extends Records
{
	protected $workBook;
	protected $workSheet;
	protected $headerStyles;
	protected $colNo = 1;
	protected $rowNo = 1;
	protected $invNo = 0;

	/**
	 * Constructor.
	 */
	public function __construct()
	{
		parent::__construct();
		$this->workBook = new Spreadsheet();
		$this->workSheet = $this->workBook->setActiveSheetIndex(0);
		$this->headerStyles = [
			'fill' => ['type' => Fill::FILL_SOLID, 'color' => ['argb' => 'E1E0F7']],
			'font' => ['bold' => true],
		];
	}

	/** {@inheritdoc} */
	public function getHeaders(): array
	{
		$headers = parent::getHeaders();
		foreach ($headers as $header) {
			$this->workSheet->setCellValueExplicit([$this->colNo, $this->rowNo], $header, DataType::TYPE_STRING);
			++$this->colNo;
		}
		++$this->rowNo;
		return $headers;
	}

	/** {@inheritDoc} */
	public function exportDataToTmpFile(): string
	{
		$headers = $this->getHeaders();

		$addInventoryData = $this->fullData && $this->moduleInstance->isInventory();
		if ($addInventoryData) {
			$inventoryModel = Vtiger_Inventory_Model::getInstance($this->moduleName);
			$inventoryFields = $inventoryModel->getFields();
			$inventoryTable = $inventoryModel->getDataTableName();
		}
		$dataReader = $this->getExportQuery()->createCommand()->query();
		while ($row = $dataReader->read()) {
			if ($addInventoryData) {
				$invRows = (new Query())->from($inventoryTable)->where(['crmid' => $row['id']])->orderBy('seq')->all();
				if ($invRows) {
					foreach ($invRows as $invRow) {
						$this->sanitizeValues($row);
						$this->sanitizeInventoryValues($invRow, $inventoryFields);
					}
				}
			} else {
				$this->sanitizeValues($row);
			}
		}
		$dataReader->close();

		return $this->saveToTmpFile($headers);
	}

	/** {@inheritdoc} */
	public function exportData()
	{
		$tmpFile = $this->exportDataToTmpFile();

		$this->output($tmpFile);
	}

	/** {@inheritdoc} */
	public function sanitizeValues(array $recordValues): array
	{
		$this->colNo = 1;
		foreach ($this->fields as $dbKey => $fieldModel) {
			$idKey = 'id';
			if ($fieldModel->get('source_field_name')) {
				/** @see \App\QueryGenerator::createColumnAlias */
				$idKey = QueryGeneratorFieldTransformer::combine($idKey, $fieldModel->getModuleName(), $fieldModel->get('source_field_name'));
				$dbKey = QueryGeneratorFieldTransformer::combine($fieldModel->getName(), $fieldModel->getModuleName(), $fieldModel->get('source_field_name'));
			}
			$this->putDataIntoSpreadsheet($fieldModel, $recordValues[$dbKey], $recordValues[$idKey] ?? 0);
		}
		++$this->rowNo;

		return [];
	}

	/**
	 * Put data into spreadsheet.
	 *
	 * @param Vtiger_Field_Model $fieldModel
	 * @param mixed              $value
	 * @param int                $id
	 *
	 * @return void
	 */
	public function putDataIntoSpreadsheet(Vtiger_Field_Model $fieldModel, $value, int $id)
	{
		switch ($fieldModel->getFieldDataType()) {
			case 'integer':
			case 'double':
			case 'currency':
				$type = is_numeric($value) ? DataType::TYPE_NUMERIC : DataType::TYPE_STRING;
				$this->workSheet->setCellValueExplicit([$this->colNo, $this->rowNo], $value, $type);
				break;
			case 'date':
				if ($value) {
					$this->workSheet->setCellValueExplicit([$this->colNo, $this->rowNo], Date::PHPToExcel($value), DataType::TYPE_NUMERIC);
					$this->workSheet->getStyle([$this->colNo, $this->rowNo])->getNumberFormat()->setFormatCode('DD/MM/YYYY');
				} else {
					$this->workSheet->setCellValueExplicit([$this->colNo, $this->rowNo], '', DataType::TYPE_STRING);
				}
				break;
			case 'datetime':
				if ($value) {
					$this->workSheet->setCellValueExplicit([$this->colNo, $this->rowNo], Date::PHPToExcel($value), DataType::TYPE_NUMERIC);
					$this->workSheet->getStyle([$this->colNo, $this->rowNo])->getNumberFormat()->setFormatCode('DD/MM/YYYY HH:MM:SS');
				} else {
					$this->workSheet->setCellValueExplicit([$this->colNo, $this->rowNo], '', DataType::TYPE_STRING);
				}
				break;
			default:
				$displayValue = $this->getDisplayValue($fieldModel, $value, $id, []) ?: '';
				$this->workSheet->setCellValueExplicit([$this->colNo, $this->rowNo], $displayValue, DataType::TYPE_STRING);
		}
		++$this->colNo;
	}

	/** {@inheritdoc} */
	public function sanitizeInventoryValues(array $inventoryRow, array $inventoryFields): array
	{
		++$this->invNo;
		$this->workSheet->setCellValueExplicit([$this->colNo, $this->rowNo], $this->invNo, DataType::TYPE_NUMERIC);
		++$this->colNo;
		foreach ($inventoryFields as $columnName => $field) {
			$value = $inventoryRow[$columnName] ?? '';
			if (\in_array($field->getType(), ['Name', 'Reference', 'Currency', 'Value', 'Unit', 'Boolean', 'Comment', 'Picklist', 'PicklistField', 'DiscountMode', 'TaxMode'])) {
				$this->workSheet->setCellValueExplicit([$this->colNo, $this->rowNo], $field->getDisplayValue($value, $inventoryRow, true), DataType::TYPE_STRING);
			} elseif ('Date' === $field->getType()) {
				if ($value) {
					$this->workSheet->setCellValueExplicit([$this->colNo, $this->rowNo], Date::PHPToExcel($value), DataType::TYPE_NUMERIC);
					$this->workSheet->getStyle([$this->colNo, $this->rowNo])->getNumberFormat()->setFormatCode('DD/MM/YYYY');
				} else {
					$this->workSheet->setCellValueExplicit([$this->colNo, $this->rowNo], '', DataType::TYPE_STRING);
				}
			} else {
				$type = is_numeric($value) ? DataType::TYPE_NUMERIC : DataType::TYPE_STRING;
				$this->workSheet->setCellValueExplicit([$this->colNo, $this->rowNo], $value, $type);
			}
			++$this->colNo;
			foreach ($field->getCustomColumn() as $customColumnName => $dbType) {
				$valueParam = $inventoryRow[$customColumnName] ?? '';
				if ('currencyparam' === $customColumnName) {
					$field = $inventoryFields['currency'];
					$valueData = $field->getCurrencyParam([], $valueParam);
					if (\is_array($valueData)) {
						$valueNewData = [];
						foreach ($valueData as $currencyId => $data) {
							$currencyName = Currency::getById($currencyId)['currency_name'];
							$valueNewData[$currencyName] = $data;
						}
						$valueParam = Json::encode($valueNewData);
					}
				}
				$this->workSheet->setCellValueExplicit([$this->colNo, $this->rowNo], $valueParam, DataType::TYPE_STRING);
				++$this->colNo;
			}
		}
		return [];
	}

	/** {@inheritdoc} */
	protected function output(string $tmpPath)
	{
		$fp = fopen($tmpPath, 'r');
		fpassthru($fp);
		fclose($fp);
	}

	/**
	 * Save export data to temporary file.
	 *
	 * @param $headers
	 *
	 * @return string
	 */
	private function saveToTmpFile($headers): string
	{
		// having written out all the data lets have a go at getting the columns to auto-size
		$row = $col = 1;
		$length = \count($headers);
		for ($i = 1; $i <= $length; ++$i) {
			$cell = $this->workSheet->getCell([$col, $row]);
			$this->workSheet->getStyle([$col, $row])->applyFromArray($this->headerStyles);
			$this->workSheet->getColumnDimension($cell->getColumn())->setAutoSize(true);
			++$col;
		}
		$tempFileName = tempnam(ROOT_DIRECTORY . DIRECTORY_SEPARATOR . Config::main('tmp_dir'), 'xls');
		$this->registerTmpFile($tempFileName);

		$workbookWriter = IOFactory::createWriter($this->workBook, ucfirst($this->fileExtension));
		$workbookWriter->save($tempFileName);

		return $tempFileName;
	}
}
