<?php
/**
 * Get related modules.
 *
 * @package   Views
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Adrian Kon <a.kon@yetiforce.com>
 */

use App\Report\Provider\RestrictedModulesProvider;
use App\User;

/**
 * Related modules view.
 */
class ReportTemplate_GetRelatedModules_View extends \App\Controller\View\Page
{
	/** {@inheritdoc} */
	public function checkPermission(App\Request $request)
	{
		if (!Users_Privileges_Model::getCurrentUserPrivilegesModel()->hasModulePermission($request->getModule())) {
			throw new \App\Exceptions\NoPermitted('LBL_PERMISSION_DENIED', 406);
		}
	}

	/** {@inheritdoc} */
	public function process(App\Request $request)
	{
		$relatedItems = $request->getArray('selectedRelatedModules', App\Purifier::ALNUM);
		$moduleRelations = $referenceModules = $advancedModules = [];
		$excludedModules = RestrictedModulesProvider::provide();
		foreach ($relatedItems as $module) {
			if (is_numeric($module)) {
				$relation = App\Relation::getById($module);
				$module = App\Module::getModuleName($relation['related_tabid']);
			}
			$moduleModel = Vtiger_Module_Model::getInstance($module);
			$referenceModules = array_merge($referenceModules, $this->getReferenceModules($moduleModel));
			$relations = $moduleModel->getRelations();
			foreach ($relations as $relationId => $relationModel) {
				$relationModuleName = $relationModel->getRelationModuleName();
				if ($module === $relationModuleName || true === in_array($relationModuleName, $excludedModules)) {
					unset($relations[$relationId]);
				}
			}
			$moduleRelations[$module] = $relations;
			$advancedModules = array_merge($advancedModules, $this->getModulesRelatedWithAdvancedBlock($moduleModel));
		}
		$allModules = [];
		if (\in_array('Users', $relatedItems)) {
			$allModules = Vtiger_Module_Model::getAll([0], [], true);
		}

		$viewer = $this->getViewer($request);
		$viewer->assign('ALL_MODULES_WHEN_USER', $allModules);
		$viewer->assign('MODULE_RELATIONS', $moduleRelations);
		$viewer->assign('REFERENCE_MODULES', $referenceModules);
		$viewer->assign('ADVANCED_MODULES', $advancedModules);
		echo $viewer->view('Modals/RelatedModules.tpl', $request->getModule(), true);
	}

	/** Get reference modules */
	private function getReferenceModules(Vtiger_Module_Model $moduleModel): array
	{
		$moduleName = $moduleModel->getName();
		$referenceModules = [];

		if (User::getCurrentUserModel()->isAdmin()) {
			$referenceModules[$moduleName][] = 'Users';
		}

		$excludedModules = RestrictedModulesProvider::provide();
		foreach ($moduleModel->getFieldsByType(array_merge(\Vtiger_Field_Model::$referenceTypes)) as $field) {
			foreach ($field->getReferenceList() as $referenceModuleName) {
				if ($referenceModuleName !== $moduleName && false === in_array($referenceModuleName, $excludedModules)) {
					$referenceModules[$moduleName][] = $referenceModuleName;
				}
			}
		}
		if(!empty(($referenceModules[$moduleName]))) {
			$referenceModules[$moduleName] = array_unique($referenceModules[$moduleName]);
		}
		return $referenceModules;
	}

	/** Get modules related with advanced block */
	private function getModulesRelatedWithAdvancedBlock(Vtiger_Module_Model $moduleModel): array
	{
		$moduleName = $moduleModel->getName();
		$relatedModules = [];
		if ($inventoryModel = $moduleModel->getInventoryModel()) {
			if ($inventoryModel->isField('name')) {
				$inventoryFieldName = $inventoryModel->getField('name');
				foreach ($inventoryFieldName->getModules() as $inventoryModuleName) {
					$relatedModules[$moduleName][] = $inventoryModuleName;
				}
			}
		}
		return $relatedModules;
	}
}
