<?php
/**
 * Common repository for db connected queries
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Łukasz Krawczyk <l.krawczyk@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Report\Repository\Template;

use App\Db\Query;
use App\Report\Builder\TemplateBuilder;
use App\Report\Model\Template;
use App\Report\Repository\TemplateRepository;

/** YiiTemplateRepository class */
final class YiiTemplateRepository implements TemplateRepository
{
	public function __construct(
		private readonly TemplateBuilder $templateBuilder,
	) {
	}

	/** {@inheritDoc} */
	public function find(int $id): Template
	{
		$result = (new Query())
			->select(
				'a.id as template_id, a.name, b.id as query_id, c.id as expression_id, d.modulename, c.field,
				c.function, c.group_by, c.order_by, c.sort, e.id as filter_group_id, e.`condition`,
				e.parent_id as filter_group_parent_id, f.id as filter_id, f.field as value_1, f.value as value_2,
				f.operator'
			)
			->from('u_yf_report_template a')
			->innerJoin('u_yf_report_template_query b', 'a.query_id = b.id')
			->innerJoin('u_yf_report_template_query_expression c', 'b.id = c.query_id')
			->innerJoin('vtiger_entityname d', 'b.base_module_id = d.tabid')
			->leftJoin('u_yf_report_template_query_filter_group e', 'b.id = e.query_id')
			->leftJoin('u_yf_report_template_query_filter f', 'e.id = f.filter_group_id')
			->andWhere(['a.id' => $id])
			->addOrderBy(['sort' => SORT_ASC, 'filter_group_id' => SORT_ASC])
			->all();

		return $this->templateBuilder->build($result);
	}

	/**
	 * Merge expression field data with YetiForce field definition
	 *
	 * @return array{field: string, label: string, moduleName: string}|null
	 */
	public function getFieldDefinition(int $id, string $fieldName, string $moduleName): ?array
	{
		$field = (new Query())
			->select(['uyrtqe.field', 'vf.fieldlabel as label', 'vt.name as moduleName'])
			->from('u_yf_report_template ufrt')
			->innerJoin('u_yf_report_template_query_expression uyrtqe', 'ufrt.query_id = uyrtqe.query_id')
			->innerJoin('vtiger_field vf', 'vf.fieldname = uyrtqe.field')
			->innerJoin('vtiger_tab vt', 'vt.tabid = vf.tabid')
			->andWhere(['ufrt.id' => $id])
			->andWhere(['uyrtqe.field' => $fieldName])
			->andWhere(['vt.name' => $moduleName])
			->one();

		return false !== $field ? $field : null;
	}
}
