<?php
/**
 * GraphHopper driver file to get coordinates.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 *
 * @see       https://www.graphhopper.com/developers/
 */

namespace App\Map\Coordinates;

/**
 * GraphHopper driver class to get coordinates.
 */
class GraphHopper extends Base
{
	/**
	 * API URL.
	 */
	private const API_URL = 'https://graphhopper.com/api/1/geocode';

	/**
	 * @inheritdoc
	 */
	protected string $label = 'LBL_COORDINATES_GRAPH_HOPPER';

	/**
	 * @inheritdoc
	 */
	protected string $docUrl = 'https://docs.graphhopper.com/#tag/Geocoding-API';

	/**
	 * @inheritdoc
	 */
	protected array $formFields = [
		'api_key' => [
			'validator' => [['name' => 'AlphaNumeric']],
			'uitype' => 99,
			'label' => 'LBL_API_KEY',
			'purifyType' => \App\Purifier::ALNUM_EXTENDED,
			'maximumlength' => '200',
			'typeofdata' => 'V~M',
		],
	];

	/**
	 * @var string[] Address params to build address query.
	 */
	private const ADDRESS_PARAMS = ['street', 'city', 'county', 'state'];

	/** {@inheritdoc} */
	public function getCoordinates(array $addressInfo, bool $onlyOne = true): ?array
	{
		if (empty($addressInfo) || !\App\RequestUtil::isNetConnection() || empty($this->getConfig()['api_key'])) {
			return null;
		}
		$url = $this->getUrl($addressInfo) . '&limit=' . ($onlyOne ? 1 : 10);

		$options = ['http_errors' => false];
		\App\Log::beginProfile("GET|GraphHopper::find|{$url}", __NAMESPACE__);
		$response = \App\RequestHttp::getClient()->get($url, $options);
		\App\Log::endProfile("GET|GraphHopper::find|{$url}", __NAMESPACE__);
		$body = $response->getBody()->getContents();
		$body = \App\Json::isEmpty($body) ? [] : \App\Json::decode($body);

		if (200 !== $response->getStatusCode() || !empty($body['message'])) {
			\App\Log::error(
				'Error: ' . $url . ' | ' . ($body['message'] ?? $response->getReasonPhrase()),
				__CLASS__
			);
			return [];
		}

		$coordinates = [];
		if ($body && !empty($body['hits'][0])) {
			foreach ($body['hits'] as $row) {
				$coordinate = [
					'lat' => $row['point']['lat'],
					'lon' => $row['point']['lng']
				];
				if ($onlyOne) {
					$coordinates = $coordinate;
					break;
				}
				$coordinates[] = $coordinate;
			}
		}
		return $coordinates;
	}

	/** {@inheritdoc} */
	public function getCoordinatesByValue(string $value): array
	{
		return $this->getCoordinates(['q' => $value]);
	}

	/**
	 * Get url to Geocoding API.
	 *
	 * @param array $addressInfo
	 *
	 * @return string
	 */
	private function getUrl(array $addressInfo): string
	{
		if (isset($addressInfo['q'])) {
			$address = $addressInfo['q'];
		} else {
			$address = '';
			foreach (self::ADDRESS_PARAMS as $value) {
				if ('' !== $addressInfo[$value]) {
					$address .= $addressInfo[$value] . ',';
				}
			}
		}
		$params = [
			'q' => rtrim($address, ','),
			'locale' => \App\Language::getShortLanguageName(),
			'key' => $this->getConfig()['api_key'],
			'reverse' => 'false'
		];
		return self::API_URL . '?' . http_build_query($params);
	}
}
