<?php
/**
 * Cleaner cli file.
 *
 * @package Cli
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

namespace App\Cli;

use App\Cache;
use App\Session;
use FilesystemIterator;
use vtlib\Functions;

/**
 * Cleaner cli class.
 */
class Cleaner extends Base
{
	/** {@inheritdoc} */
	public string $moduleName = 'Cleaner';

	/** {@inheritdoc} */
	protected array $methods = [
		'logs' => 'Delete all logs',
		'session' => 'Delete all sessions',
		'cacheData' => 'Cache data',
		'cacheFiles' => 'Cache files',
	];

	/**
	 * Delete all logs.
	 *
	 * @return void
	 */
	public function logs(): void
	{
		$i = $s = 0;
		$this->climate->bold('Removing all logs...');
		$iterator = new \RecursiveIteratorIterator(
			new \RecursiveDirectoryIterator(ROOT_DIRECTORY . '/cache/logs', FilesystemIterator::SKIP_DOTS),
			\RecursiveIteratorIterator::SELF_FIRST
		);
		foreach ($iterator as $item) {
			if ($item->isFile() && 'index.html' !== $item->getBasename()) {
				$this->climate->bold($iterator->getSubPathName() . ' - ' . Functions::showBytes($item->getSize()));
				$s += $item->getSize();
				unlink($item->getPathname());
				++$i;
			}
		}
		$this->climate->lightYellow()->border('─', 200);
		$this->climate->bold('Number of deleted log files: ' . $i);
		$this->climate->bold('Size of deleted log files: ' . Functions::showBytes($s));
		$this->climate->lightYellow()->border('─', 200);
	}

	/**
	 * Delete all session.
	 *
	 * @return void
	 */
	public function session(): void
	{
		$this->climate->bold('Removing all sessions...');
		Session::load();
		$this->climate->bold('Number of deleted sessions: ' . Session::cleanAll());
		$this->climate->lightYellow()->border('─', 200);
		if (!$this->climate->arguments->defined('action')) {
			$this->returnToActionList();
		}
	}

	/**
	 * Clear cache data.
	 *
	 * @return void
	 */
	public function cacheData(): void
	{
		$this->climate->bold('Clear: ' . Cache::clear());
		$this->climate->bold('Clear opcache: ' . Cache::clearOpcache());
		$this->climate->lightYellow()->border('─', 200);
		if (!$this->climate->arguments->defined('action')) {
			$this->returnToActionList();
		}
	}

	/**
	 * Clear cache files.
	 *
	 * @return void
	 */
	public function cacheFiles(): void
	{
		$stats = Cache::clearTemporaryFiles('now');
		$this->climate->bold(" - files: {$stats['counter']} , size: " . Functions::showBytes($stats['size']));
		$this->climate->lightYellow()->border('─', 200);
		if (!$this->climate->arguments->defined('action')) {
			$this->returnToActionList();
		}
	}
}
