<?php
/**
 * Project module model class.
 *
 * @package   Module
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 */

use App\Db\Query;
use App\Record;
use yii\db\Expression;

class ProjectMilestone_Module_Model extends Vtiger_Module_Model
{
	/**
	 * Update progress in project milestone.
	 *
	 * @param int $id
	 *
	 * @return void
	 */
	public static function updateProgress(int $id): void
	{
		if (!Record::isExists($id)) {
			return;
		}

		$recordModel = Vtiger_Record_Model::getInstanceById($id);
		$progressField = $recordModel->getField('projectmilestone_progress');
		$estimatedField = $recordModel->getField('estimated_work_time');
		if ((!$progressField || !$progressField->isActiveField()) && (!$estimatedField || !$estimatedField->isActiveField())) {
			return;
		}

		$estimatedWorkTime = 0;
		$progressInHours = 0;
		$pmIds = static::getChildren($id, [$id]);

		foreach (array_chunk($pmIds, 20) as $partPMIds) {
			$row = (new Query())
				->select([
					'estimated_work_time' => new Expression('SUM(vtiger_projecttask.estimated_work_time)'),
					'progress_in_hours' => new Expression('SUM(vtiger_projecttask.estimated_work_time * vtiger_projecttask.projecttaskprogress / 100)')
				])
				->from('vtiger_projecttask')
				->innerJoin('vtiger_crmentity', 'vtiger_projecttask.projecttaskid = vtiger_crmentity.crmid')
				->where(['vtiger_projecttask.projectmilestoneid' => $partPMIds])
				->andWhere(['vtiger_crmentity.deleted' => [0, 2]])
				->one();

			$estimatedWorkTime += (float) ($row['estimated_work_time'] ?? 0);
			$progressInHours += (float) ($row['progress_in_hours'] ?? 0);
		}

		$projectProgress = $estimatedWorkTime ? round(($progressInHours * 100) / $estimatedWorkTime) : 0;

		$recordModel->set($progressField->getName(), $projectProgress);
		$recordModel->setDataForSave([$progressField->getTableName() => [$progressField->getColumnName() => $projectProgress]]);

		$recordModel->set($estimatedField->getName(), $estimatedWorkTime);
		$recordModel->setDataForSave([$estimatedField->getTableName() => [$estimatedField->getColumnName() => $estimatedWorkTime]]);

		if (false !== $recordModel->getPreviousValue($progressField->getName()) || false !== $recordModel->getPreviousValue($estimatedField->getName())) {
			$recordModel->save();

			$parentPmId = $recordModel->get('parentid');
			if (!empty($parentPmId)) {
				static::updateProgress($parentPmId);
			}
		}
	}

	/**
	 * Function to get list view query for popup window.
	 *
	 * @param Vtiger_ListView_Model $listviewModel
	 * @param \App\QueryGenerator   $queryGenerator
	 */
	public function getQueryByRelatedField(Vtiger_ListView_Model $listviewModel, App\QueryGenerator $queryGenerator)
	{
		if ('Project' === $listviewModel->get('src_module') && !$listviewModel->isEmpty('filterFields')) {
			$filterFields = $listviewModel->get('filterFields');
			if (!empty($filterFields['projectid'])) {
				$queryGenerator->addNativeCondition(['projectid' => (int) $filterFields['projectid']]);
			}
		}
	}

	/**
	 * Get child element IDs by hierarchy.
	 *
	 * @param int   $id
	 * @param array $pmIds
	 *
	 * @return array
	 */
	private static function getChildren(int $id, array $pmIds = []): array
	{
		$ids = (new Query())
			->select(['id' => 'vtiger_projectmilestone.projectmilestoneid'])
			->from('vtiger_projectmilestone')
			->innerJoin('vtiger_crmentity', 'vtiger_projectmilestone.projectmilestoneid = vtiger_crmentity.crmid')
			->where(['vtiger_projectmilestone.parentid' => $id])
			->andWhere(['vtiger_crmentity.deleted' => [0, 2]])
			->column();

		$ids = array_diff($ids, $pmIds);
		$pmIds = array_merge($pmIds, $ids);

		foreach ($ids as $pmId) {
			$pmIds = static::getChildren($pmId, $pmIds);
		}

		return $pmIds;
	}
}
