<?php

/**
 * OSSMailScanner Record model class.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
class OSSMailScanner_Record_Model extends Vtiger_Record_Model
{
	/**
	 * Active users list.
	 *
	 * @var array|bool
	 */
	protected $user = false;

	/**
	 * Groups list.
	 *
	 * @var array
	 */
	protected $group = false;

	/**
	 * Return email search results.
	 *
	 * @param string $module
	 * @param mixed  $onlyMailUitype
	 *
	 * @return array
	 */
	public static function getEmailSearch($module = false, $onlyMailUitype = true)
	{
		$return = [];
		$query = (new App\Db\Query())->from('vtiger_field')
			->leftJoin('vtiger_tab', 'vtiger_tab.tabid = vtiger_field.tabid')
			->where(['and', ['uitype' => ($onlyMailUitype ? 13 : [13, 319])], ['<>', 'vtiger_field.presence', 1], ['<>', 'vtiger_tab.name', 'Users']]);
		if ($module) {
			$query->andWhere(['vtiger_tab.name' => $module]);
		}
		$query->orderBy('name');
		$dataReader = $query->createCommand()->query();
		while ($row = $dataReader->read()) {
			$return[] = [
				'key' => "{$row['fieldname']}={$row['name']}={$row['uitype']}",
				'value' => "{$row['fieldname']}={$row['name']}", // item to delete in next version
				'fieldlabel' => $row['fieldlabel'],
				'tablename' => $row['tablename'],
				'columnname' => $row['columnname'],
				'name' => $row['name'],
				'tabid' => $row['tabid'],
				'fieldname' => $row['fieldname'],
			];
		}
		$dataReader->close();
		return $return;
	}

	/**
	 * Return history status label.
	 *
	 * @param int $id
	 *
	 * @return string
	 */
	public function getHistoryStatus($id)
	{
		switch ($id) {
			case 0:
				$return = 'OK';
				break;
			case 1:
				$return = 'In progress';
				break;
			case 2:
				$return = 'Manually stopped';
				break;
			case 3:
				$return = 'Error';
				break;
			default:
				break;
		}
		return $return;
	}

	/**
	 * Return scan history.
	 *
	 * @param int $startNumber
	 *
	 * @return array
	 */
	public function getScanHistory($startNumber = 0)
	{
		$limit = 30;
		$endNumber = $startNumber + $limit;
		$dataReader = (new App\Db\Query())->from('vtiger_ossmails_logs')->orderBy(['id' => SORT_DESC])->limit($endNumber)->offset($startNumber)->createCommand()->query();
		$output = [];
		while ($row = $dataReader->read()) {
			$startTime = new DateTimeField($row['start_time']);
			$endTime = new DateTimeField($row['end_time']);
			$output[] = [
				'id' => $row['id'],
				'start_time' => $startTime->getDisplayDateTimeValue(),
				'end_time' => $endTime->getDisplayDateTimeValue(),
				'status' => self::getHistoryStatus($row['status']),
				'user' => $row['user'],
				'stop_user' => $row['stop_user'],
				'count' => $row['count'],
				'action' => $row['action'],
				'info' => $row['info'],
			];
		}
		$dataReader->close();

		return $output;
	}

	/**
	 * Insert new scan history row.
	 *
	 * @param array $array
	 *
	 * @throws \yii\db\Exception
	 *
	 * @return int
	 */
	public function addScanHistory($array): int
	{
		$db = \App\Db::getInstance();
		$db->createCommand()->insert('vtiger_ossmails_logs', ['status' => 1, 'user' => $array['user'], 'start_time' => date('Y-m-d H:i:s')])->execute();

		return $db->getLastInsertID('vtiger_ossmails_logs_id_seq');
	}

	/**
	 * Update scan history row.
	 *
	 * @param int   $id
	 * @param array $array
	 */
	public static function updateScanHistory($id, $array): bool
	{
		$dbCommand = \App\Db::getInstance()->createCommand();
		$result = $dbCommand->update(
			'vtiger_ossmails_logs',
			['end_time' => date('Y-m-d H:i:s'), 'status' => $array['status'], 'count' => $array['count'], 'action' => $array['action']],
			['and', ['id' => $id], ['<>', 'status', 2]]
		)->execute();
		if (!$result) {
			$dbCommand->update('vtiger_ossmails_logs', ['count' => $array['count']], ['id' => $id])->execute();
		}
		return (bool) $result;
	}

	/**
	 * State of scan action.
	 *
	 * @param int|null $scanId
	 *
	 * @return bool
	 */
	public static function isActiveScan(?int $scanId = null): bool
	{
		$where = ['status' => 1];
		if ($scanId) {
			$where['id'] = $scanId;
		}
		return (new App\Db\Query())->select(['id'])->from('vtiger_ossmails_logs')->where($where)->exists();
	}

	/**
	 * Activate scan.
	 *
	 * @param string   $whoTrigger
	 * @param int|null $scanId
	 *
	 * @return bool
	 */
	public static function setActiveScan(string $whoTrigger, ?int $scanId = null): bool
	{
		$where = ['status' => 1];
		if ($scanId) {
			$where['id'] = $scanId;
		}
		return (bool) \App\Db::getInstance()->createCommand()
			->update('vtiger_ossmails_logs', ['status' => 2, 'stop_user' => $whoTrigger, 'end_time' => date('Y-m-d H:i:s')], $where)->execute();
	}

	/**
	 * Cron data.
	 *
	 * @return \vtlib\Cron
	 */
	public static function getCronTask()
	{
		return \vtlib\Cron::getInstance('LBL_MAIL_SCANNER_ACTION')->refreshData();
	}

	/**
	 * Restart cron.
	 *
	 * @param int $scanId
	 */
	public static function runRestartCron(int $scanId)
	{
		if (self::isActiveScan($scanId)) {
			if (self::getCronTask()->hadTimeout()) {
				\App\Cron::updateStatus(\App\Cron::STATUS_ENABLED, 'LBL_MAIL_SCANNER_ACTION');
			}
			self::setActiveScan(\App\User::getCurrentUserModel()->getDetail('user_name'), $scanId);
		}
	}
}
