<?php
/**
 * Mail scanner file.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

namespace App\Mail;

/**
 * Mail scanner class.
 */
class Scanner extends \App\Base
{
	/** @var string Base module name */
	public const MODULE_NAME = 'MailAccount';
	/** @var string Name of the scanned folder table */
	public const FOLDER_TABLE = 'u_#__mailscanner_folders';
	/** @var string Logs table name */
	public const LOG_TABLE = 'vtiger_ossmails_logs';

	/** @var \App\Mail\Account */
	private $account;
	/** @var int Limit of scanned e-mails */
	private int $limit = 100;
	/** @var int Number of scanned e-mails */
	private $count = 0;
	/** @var ScannerLog Scanner log */
	private $log;

	/**
	 * Constructor.
	 */
	public function __construct()
	{
		$this->log = new ScannerLog();
	}

	/**
	 * Set limit.
	 *
	 * @param int $limit
	 *
	 * @return $this
	 */
	public function setLimit(int $limit)
	{
		$this->limit = $limit;
		return $this;
	}

	/**
	 * Returns the number of scanned emails.
	 *
	 * @return int
	 */
	public function getCount(): int
	{
		return $this->count;
	}

	/**
	 * Set Mail Account.
	 *
	 * @param Account $account
	 *
	 * @return $this
	 */
	public function setAccount(Account $account)
	{
		$this->account = $account;
		return $this;
	}

	/**
	 * Get last UID.
	 *
	 * @param string $folderName
	 *
	 * @return int|null
	 */
	public function getLastUid(string $folderName)
	{
		return (new \App\Db\Query())->select(['uid'])->from(self::FOLDER_TABLE)
			->where(['id' => $this->account->getSource()?->getId() ?? 0, 'name' => $folderName])->scalar();
	}

	/**
	 * Set UID to folder.
	 *
	 * @param int    $uid
	 * @param string $folderName
	 *
	 * @return bool
	 */
	public function setLastUid(int $uid, string $folderName): bool
	{
		return (bool) \App\Db::getInstance()->createCommand()
			->update(self::FOLDER_TABLE, ['uid' => $uid], ['id' => $this->account->getSource()?->getId() ?? 0, 'name' => $folderName])->execute();
	}

	/**
	 * Run scanner.
	 *
	 * @param callable|null $callback Conditions for scanning subsequent emails
	 *
	 * @return void
	 */
	public function run(?callable $callback = null)
	{
		$folders = $this->account->getFolders();
		$actions = $this->account->getActions();
		if ($folders && $actions && ($imap = $this->getImapConnection()) && ($folders = array_intersect($folders, array_keys($imap->getFolders(false))))) {
			$flagSeen = \App\Mail::getConfig('scanner', 'flag_seen');
			foreach ($folders as $folderName) {
				$this->log->start();
				$uid = $this->getLastUid($folderName);
				if (!\is_int($uid)) {
					continue;
				}

				/* No emails , @see https://github.com/Webklex/php-imap/issues/499 * */
				if (!($imap->getFolderByName($folderName)->examine()['exists'] ?? false)) {
					continue;
				}

				$messageCollection = $imap->getMessagesGreaterThanUid($folderName, $uid, $this->limit);
				foreach ($messageCollection as $message) {
					if (!$this->log->isRunning()) {
						return;
					}
					if ($callback && $callback($this)) {
						break 2;
					}
					--$this->limit;
					$messageObject = (new Message\Imap())->setMessage($message);
					try {
						foreach ($actions as $action) {
							$this->getAction($action)->setAccount($this->account)->setMessage($messageObject)->process();
						}
						++$this->count;
						$this->setLastUid($messageObject->getMsgUid(), $folderName);
						$this->log->updateCount($this->count);
						if ($flagSeen) {
							$messageObject->setFlag('Seen');
						}
					} catch (\Throwable $th) {
						$message = $th->getMessage();
						\App\Log::error("Mail Scanner - Account: {$this->account->getSource()->getId()}, folder: {$folderName}, UID: {$messageObject->getMsgUid()}, message: " . $message);
						if ($th instanceof \App\Exceptions\AppException) {
							$message = $th->getDisplayMessage();
						}
						$this->log->close(ScannerLog::STATUS_ERROR, $message, $action);
						break;
					}
				}

				if (0 === $this->limit) {
					break;
				}
			}

			$this->log->close();
		}
	}

	/**
	 * Check if the limit has been reached.
	 *
	 * @return bool
	 */
	public function isLimitReached(): bool
	{
		return 0 === $this->limit;
	}

	/**
	 * Get imap connection.
	 *
	 * @return Connections\Imap|null
	 */
	public function getImapConnection(): ?Connections\Imap
	{
		try {
			$imap = $this->account->openImap();
			if ($imap->isConnected()) {
				$this->account->getSource()->setLogs('');
				$this->account->update(['logs']);
			}
		} catch (\Throwable $th) {
			$this->account->lock($th->getMessage());
			$imap = null;
		}

		return $imap;
	}

	/**
	 * Get action object.
	 *
	 * @param string $name
	 *
	 * @return ScannerAction\Base
	 */
	public function getAction(string $name): ScannerAction\Base
	{
		$class = "App\\Mail\\ScannerAction\\{$name}";
		return new $class();
	}

	/**
	 * Check if ready.
	 *
	 * @return bool
	 */
	public function isReady(): bool
	{
		return !ScannerLog::isScannRunning();
	}
}
